<?php
/**
 * Akeeba Backup Restoration Script
 *
 * @package   brs
 * @copyright Copyright (c)2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\BRS\Platform\Model;

defined('_AKEEBA') or die();

use Akeeba\BRS\Framework\Database\DatabaseDriverInterface;
use Akeeba\BRS\Model\AbstractConfiguration;
use Exception;
use Psr\Container\ContainerInterface;

/**
 * WordPress configuration handling model.
 *
 * @since  10.0
 */
class Configuration extends AbstractConfiguration
{
	/** @inheritDoc */
	public function __construct(?ContainerInterface $container = null, array $config = [])
	{
		$this->configFilename = 'wp-config.php';

		// Call the parent constructor
		parent::__construct($container, $config);

		// Load the configuration variables from the session.
		$this->configvars = (array) ($this->getContainer()->get('session')->get('cms.config') ?: []);

		// Fall back to loading from disk
		if (empty($this->configvars) || empty($this->configvars['blogname']))
		{
			$this->reloadConfiguration();
		}

		// Change the default charset and collation if UTF8MB4 is enabled.
		if ($this->supportsUTF8MB4() && $this->get('dbcharset') === 'utf8' && $this->get('dbcollation') == '')
		{
			$this->set('dbcharset', 'utf8mb4');
			$this->set('dbcollation', 'utf8mb4_unicode_ci');
		}
	}

	/**
	 * Returns an associative array with default settings
	 *
	 * @return  array
	 * @since   10.0
	 */
	public function getDefaultConfig(): array
	{
		return [
			// Status flag
			'readConfigFromDisk' => false,
			// MySQL settings
			'dbname'             => '',
			'dbuser'             => '',
			'dbpass'             => '',
			'dbhost'             => '',
			'dbcharset'          => '',
			'dbcollation'          => '',
			'dbprefix'           => '',
			// Other
			'blogname'           => '',
		];
	}

	/**
	 * Try to parse the wp-config.php file.
	 *
	 * WordPress' wp-config.php file is a plain old PHP file with some define statements. Unfortunately, its free
	 * form nature means that any changes made by site owners and/or hosts can lead to problems if we try to
	 * include it directly – even if we take into account that it tries to load more of WordPress itself at the very
	 * end of the file. The worst happens when people use conditionals, environment variables, control structures,
	 * and the like.
	 *
	 * The only option we have is to parse it line by line, trying to extract the lines with the define statements.
	 *
	 * @param   string  $file  The full path to the file
	 *
	 * @return  array
	 * @since   10.0
	 */
	public function loadFromFile(string $file): array
	{
		$config   = [];
		$contents = @file_get_contents($file) ?: '';

		/**
		 * Remove comment lines.
		 *
		 * If the PHP tokenizer extension is present, we will use it to remove comments. This is the preferred way of
		 * doing it. Otherwise, we will use a rather precarious Regular Expression to do the same.
		 *
		 * In the latter case we will end up stripping all lines with comments, even comments after assigment, like
		 * $foo = 'bar' #same line comment
		 * This will invalidate the Authentication Keys and Salt part, but that's not a problem since we will have to
		 * ultimately change them.
		 *
		 * @link http://stackoverflow.com/a/13114141/485241
		 */
		$contents = function_exists('token_get_all')
			? $this->stripComments($contents)
			: preg_replace(
				'~(?:#|//)[^\r\n]*|/\*.*?\*/~s', '', $contents
			);

		// Process the file line by line.
		$lines = explode("\n", $contents);

		foreach ($lines as $line)
		{
			$line = trim($line);

			// Search for defines
			if (strpos($line, 'define') === 0)
			{
				$line = substr($line, 6);
				$line = trim($line);
				$line = rtrim($line, ';');
				$line = trim($line);
				$line = trim($line, '()');
				[$key, $value] = explode(',', $line);
				$key   = trim($key);
				$key   = trim($key, "'\"");
				$value = trim($value);
				$value = trim($value, "'\"");

				switch (strtoupper($key))
				{
					case 'DB_NAME':
						$config['dbname'] = $value;
						break;

					case 'DB_USER':
						$config['dbuser'] = $value;
						break;

					case 'DB_PASSWORD':
						$config['dbpass'] = $value;
						break;

					case 'DB_HOST':
						$config['dbhost'] = $value;
						break;

					case 'DB_CHARSET':
						$config['dbcharset'] = $value;
						break;

					case 'DB_COLLATE':
						$config['dbcollation'] = $value;
						break;

					case 'DOMAIN_CURRENT_SITE':
						$config['domain_current_site'] = $value;
						break;

					case 'PATH_CURRENT_SITE':
						$config['path_current_site'] = $value;
						break;

					case 'SITE_ID_CURRENT_SITE':
						$config['site_id_current_site'] = $value;
						break;

					case 'BLOG_ID_CURRENT_SITE':
						$config['blog_id_current_site'] = $value;
						break;

					case 'MULTISITE':
						switch (strtoupper($value))
						{
							case 'FALSE':
							case '0':
								$value = false;
								break;

							default:
								$value = true;
								break;
						}

						$config['multisite'] = $value;
						break;

					case 'SUBDOMAIN_INSTALL':
						switch (strtoupper($value))
						{
							case 'FALSE':
							case '0':
								$value = false;
								break;

							default:
								$value = true;
								break;
						}

						$config['subdomain_install'] = $value;
						break;
				}
			}
			// Table prefix
			elseif (strpos($line, '$table_prefix') === 0)
			{
				$parts      = explode('=', $line, 2);
				$prefixData = trim($parts[1]);
				$prefixData = rtrim($prefixData, ';');
				$prefixData = trim($prefixData, "'\"");

				$config['olddbprefix'] = $prefixData;
				$config['dbprefix']    = $prefixData;
			}
			// Base directory = $base
			elseif (strpos($line, '$base') === 0)
			{
				$parts      = explode('=', $line, 2);
				$prefixData = trim($parts[1]);
				$prefixData = rtrim($prefixData, ';');
				$prefixData = trim($prefixData, "'\"");

				$config['base'] = $prefixData;
			}

		}

		return $config;
	}

	/**
	 * Creates the string that will be put inside the new configuration file.
	 *
	 * This is a separate function, so we can show the content if we're unable to write to the filesystem
	 * and ask the user to manually do that.
	 *
	 * @param   string|null  $file  The wp-config.php file to read from.
	 *
	 * @return string
	 * @since  10.0
	 */
	public function getFileContents(?string $file = null): string
	{
		$paths = $this->getContainer()->get('paths');

		if (!$file)
		{
			$file = $paths->get('root') . '/wp-config.php';
		}

		$new_config = '';
		$old_config = @file_get_contents($file) ?: '';

		// Check if the file is UTF encoded with BOM. We have to remove it or we will get a white page
		// Sadly several editors are setting the flag automatically; since they are not visible, the user has
		// no easy method to remove them
		$bom = pack("CCC", 0xef, 0xbb, 0xbf);

		if (strncmp($old_config, $bom, 3) === 0)
		{
			// Let's strip out any BOM char
			$old_config = substr($old_config, 3);
		}

		$lines = explode("\n", $old_config);

		foreach ($lines as $line)
		{
			$line    = trim($line);
			$matches = [];

			// Skip commented lines. However it will get the line between a multiline comment, but that's not a problem
			/** @noinspection PhpStatementHasEmptyBodyInspection */
			if (strpos($line, '#') === 0 || strpos($line, '//') === 0 || strpos($line, '/*') === 0)
			{
				// simply do nothing, we will add the line later
			}
			elseif (strpos($line, 'define(') !== false)
			{
				preg_match('#define\(\s?["\'](.*?)["\']\,#', $line, $matches);

				if (isset($matches[1]))
				{
					$key = $matches[1];

					switch (strtoupper($key))
					{
						case 'DB_NAME' :
							$value = $this->get('dbname');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'DB_USER':
							$value = $this->get('dbuser');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'DB_PASSWORD':
							$value = $this->get('dbpass');
							$value = addcslashes($value, "'\\");
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'DB_HOST':
							$value = $this->get('dbhost');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'DB_CHARSET':
							$value = $this->get('dbcharset', $this->supportsUTF8MB4() ? 'utf8mb4' : 'utf8');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'DB_COLLATE':
							$value = $this->get('dbcollation', $this->supportsUTF8MB4() ? 'utf8mb4_unicode_ci' : 'utf8_general_ci');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'AUTH_KEY':
							$value = $this->get('auth_key');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'SECURE_AUTH_KEY':
							$value = $this->get('secure_auth_key');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'LOGGED_IN_KEY':
							$value = $this->get('logged_in_key');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'NONCE_KEY':
							$value = $this->get('nonce_key');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'AUTH_SALT':
							$value = $this->get('auth_salt');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'SECURE_AUTH_SALT':
							$value = $this->get('secure_auth_salt');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'LOGGED_IN_SALT':
							$value = $this->get('logged_in_salt');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						case 'NONCE_SALT':
							$value = $this->get('nonce_salt');
							$line  = "define('" . $key . "', '" . $value . "');";
							break;

						// Multisite variable - Main site's domain
						case 'DOMAIN_CURRENT_SITE':
							$new_url   = $this->get('homeurl');
							$newUri    = $this->getContainer()->get('uri')->instance($new_url);
							$newDomain = $newUri->getHost();
							$line      = "define('" . $key . "', '" . $newDomain . "');";
							break;

						// Multisite variable - Main site's path
						case 'PATH_CURRENT_SITE':
							$newPath = $this->getNewBasePath();
							$line    = "define('" . $key . "', '" . $newPath . "');";
							break;

						case 'WP_HOME':
							$line = "define('" . $key . "', '" . $this->get('homeurl') . "');";
							break;

						case 'WP_SITEURL':
							$line = "define('" . $key . "', '" . $this->get('siteurl') . "');";
							break;

						case 'SUBDOMAIN_INSTALL':
							/**
							 * We have a subdomain installation if
							 * - the existing site was a subdomain multisite installation; AND
							 * - we are NOT converting to a subdirectory format
							 */
							$isSubdomainInstall = $this->get('subdomain_install', 0);
							$line               = "define('" . $key . "', " . ($isSubdomainInstall ? 'true' : 'false')
							                      . ");";
							break;

						// 3rd party extensions
						case 'WPCACHEHOME':
							// WP Super Cache stores the absolute path. Let's blank it out so it will auto fix on the next load
							$line = "define('" . $key . "', '');";
							break;
						// I think users shouldn't change the WPLANG define, since they will have
						// to add several files, it's not automatic
						default:
							// Do nothing, it's a variable we're not interested in
							break;
					}
				}
			}
			elseif (strpos($line, '$table_prefix') === 0)
			{
				$line = '$table_prefix = ' . "'" . $this->get('dbprefix') . "';";
			}
			elseif (strpos($line, '$base') === 0)
			{
				$line = '';
				$base = $this->get('base', '');

				if (!empty($base))
				{
					$base = str_replace('\'', '\\\'', $base);
					$line = '$base= ' . "'" . $base . "';";
				}

			}

			$new_config .= $line . "\n";
		}

		// Temporarily remove the end of line from the last line of the file
		$new_config = rtrim($new_config);

		// Remove a closing PHP tag from the file BECAUSE IT'S BAD PRACTICE TO HAVE ONE
		if (substr($new_config, -2) == '?>')
		{
			$new_config = substr($new_config, 0, -2);
		}

		// Finally, add a newline before EOF
		$new_config .= "\n";

		return $new_config;
	}

	/**
	 * Writes the new config params inside the wp-config.php file and the database.
	 *
	 * @param   string  $file
	 *
	 * @return  bool
	 * @since   10.0
	 */
	public function writeConfig(string $file): bool
	{
		return (bool) (@file_put_contents($file, $this->getFileContents($file)));
	}

	/**
	 * Add options read from the database into the array of configuration variables.
	 *
	 * @param   array  $config
	 *
	 * @return  void
	 * @since   10.0
	 */
	public function addOptionsFromDatabase(array &$config): void
	{
		try
		{
			$db = $this->getSiteDatabase();

			$searchFor = [
				$db->q('blogname'),
				$db->q('blogdescription'),
				$db->q('home'),
				$db->q('siteurl'),
			];

			$query      = $db->getQuery(true)
				->select([$db->qn('option_name'), $db->qn('option_value')])
				->from('#__options')
				->where($db->qn('option_name') . ' IN (' . implode(',', $searchFor) . ')');
			$wp_options = $db->setQuery($query)->loadObjectList();

			foreach ($wp_options as $option)
			{
				// Let me save the old home url, it will be useful later when I'll have to replace it inside the posts
				if ($option->option_name == 'home')
				{
					$config['oldurl'] = $option->option_value;
				}
				else
				{
					$config[$option->option_name] = $option->option_value;
				}
			}
		}
		catch (Exception $exc)
		{
		}
	}

	/**
	 * Get the path portion of homeurl, WITH a leading slash, WITHOUT a trailing slash
	 *
	 * @return  string
	 * @since   10.0
	 */
	public function getNewBasePath(): string
	{
		$new_url = $this->get('homeurl');
		$newUri  = $this->getContainer()->get('uri')->instance($new_url);
		$newPath = $newUri->getPath();
		$newPath = trim($newPath, '/');
		$newPath = empty($newPath) ? '/' : '/' . $newPath . '/';

		return $newPath;
	}

	/**
	 * @inheritDoc
	 */
	public function __toString()
	{
		return $this->getFileContents();
	}

	/**
	 * Get a database object for the site's database.
	 *
	 * @return  DatabaseDriverInterface
	 * @since   10.0
	 */
	private function getSiteDatabase(): DatabaseDriverInterface
	{
		static $db;

		if ($db)
		{
			return $db;
		}

		$model          = $this->getContainer()->get('mvcFactory')->model('Database');
		$keys           = $model->getDatabaseNames();
		$firstDbKey     = array_shift($keys);
		$connectionVars = $model->getDatabaseInfo($firstDbKey);

		$name    = $connectionVars->dbtype;
		$options = [
			'database' => $connectionVars->dbname,
			'select'   => 1,
			'host'     => $connectionVars->dbhost,
			'user'     => $connectionVars->dbuser,
			'password' => $connectionVars->dbpass,
			'prefix'   => $connectionVars->prefix,
		];

		return $db = $this->getContainer()->get('db')->driver($name, $options);
	}

	/**
	 * Does the site database support UTF8MB4?
	 *
	 * Return true if detection of UTF8MB4 support is enabled, and the database supports UTF8MB4.
	 *
	 * @return  bool
	 * @since   10.0
	 */
	private function supportsUTF8MB4(): bool
	{
		$dbConfigs = $this->getContainer()->get('configuration')->databases;
		$dbConfig  = reset($dbConfigs);

		if (!$dbConfig->utf8mb4)
		{
			return false;
		}

		$db = $this->getSiteDatabase();

		return $db->hasUTFSupport() && $db->supportsUtf8mb4();
	}

	/**
	 * Use the PHP Tokenizer extension to strip comments from a chunk of PHP code.
	 *
	 * @param   string|null  $fileContents  The PHP code to clean up.
	 *
	 * @return  string|null
	 * @since   10.0
	 */
	private function stripComments(?string $fileContents): ?string
	{
		$tokens = token_get_all($fileContents ?? '');

		$commentTokens = [T_COMMENT];

		if (defined('T_DOC_COMMENT'))
		{
			$commentTokens[] = T_DOC_COMMENT;
		}

		if (defined('T_ML_COMMENT'))
		{
			/** @noinspection PhpUndefinedConstantInspection */
			$commentTokens[] = T_ML_COMMENT;
		}

		$newStr = '';

		foreach ($tokens as $token)
		{
			if (is_array($token))
			{
				if (in_array($token[0], $commentTokens))
				{
					/**
					 * If the comment ended in a newline we need to output the newline. Otherwise we will have
					 * run-together lines which won't be parsed correctly by parseWithoutTokenizer.
					 */
					if (substr($token[1], -1) == "\n")
					{
						$newStr .= "\n";
					}

					continue;
				}

				$token = $token[1];
			}

			$newStr .= $token;
		}

		return $fileContents;
	}

	/**
	 * Reloads the configuration from wp-config.php, and from the database.
	 *
	 * @return  void
	 * @since   10.0
	 */
	private function reloadConfiguration(): void
	{
		$this->configvars = $this->getDefaultConfig();
		$realConfig       = [];
		$paths            = $this->getContainer()->get('paths');

		if (!$this->configvars['readConfigFromDisk'])
		{
			$realConfig = $this->loadFromFile($paths->get('configuration') . '/wp-config.php');

			$this->addOptionsFromDatabase($realConfig);
			$this->configvars['readConfigFromDisk'] = true;
		}

		$this->configvars = array_merge($this->configvars, $realConfig);

		if (!empty($this->configvars))
		{
			$this->saveToSession();
		}
	}
}