<?php
/**
 * Akeeba Backup Restoration Script
 *
 * @package   brs
 * @copyright Copyright (c)2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\BRS\Platform\View\Setup;

defined('_AKEEBA') or die();

use Akeeba\BRS\Framework\Helper\Select;
use Akeeba\BRS\Framework\Mvc\View;
use Akeeba\BRS\Platform\Model\Setup;
use Akeeba\BRS\View\ShowOnTrait;
use Akeeba\BRS\View\StepsTrait;

/**
 * View Controller for the Site Setup step for WordPress sites.
 *
 * @since  10.0
 */
class Html extends View
{
	use StepsTrait;
	use ShowOnTrait;

	/**
	 * Site configuration variables to communicate to the frontend.
	 *
	 * @var   object
	 * @since 10.0
	 */
	public $stateVars;

	/**
	 * Checkbox options for auto-prepend handling.
	 *
	 * @var   array
	 * @since 10.0
	 */
	public $auto_prepend = [];

	/**
	 * FALSE if auto-prepend disabled, path to config file enabling auto-prepend otherwise.
	 *
	 * @var   bool|string
	 * @since 10.0
	 */
	public $hasAutoPrepend = false;

	/**
	 * Options for .htpasswd handling.
	 *
	 * @var   array
	 * @since 10.0
	 */
	public $removeHtpasswdOptions = [];

	/**
	 * Does the site have a .htaccess file?
	 *
	 * @var   bool
	 * @since 10.0
	 */
	public $hasHtaccess = false;

	/**
	 * Which .htaccess handling option is currently selected.
	 *
	 * @var   string
	 * @since 10.0
	 */
	public $htaccessOptionSelected = 'none';

	/**
	 * Options for .htaccess handling.
	 *
	 * @var   array
	 * @since 10.0
	 */
	public $htaccessOptions = [];

	/**
	 * Flag to signal the absence of any Administrator / Super Administrator account.
	 *
	 * @var   bool
	 * @since 10.0
	 */
	public $noSuper = false;

	public function onBeforeMain(): bool
	{
		$doc  = $this->getContainer()->get('application')->getDocument();
		$lang = $this->getContainer()->get('language');

		/** @var Setup $model */
		$model = $this->getModel();

		$this->stateVars      = $model->getStateVariables();
		$this->hasAutoPrepend = $model->hasAutoPrepend();
		$this->noSuper        = empty($this->stateVars->superusers ?? []);

		// Prime the options array with some default info
		$this->auto_prepend = [
			'checked'  => '',
			'disabled' => '',
		];

		$this->removeHtpasswdOptions = [
			'checked'  => '',
			'disabled' => '',
			'help'     => 'SETUP_LBL_SERVERCONFIG_REMOVEHTPASSWD_HELP',
		];

		// If we are restoring to a new server everything is checked by default
		if ($model->isNewhost())
		{
			$this->auto_prepend['checked']          = 'checked="checked"';
			$this->removeHtpasswdOptions['checked'] = 'checked="checked"';
		}

		/**
		 * If any option is not valid (ie missing files) we gray out the option AND remove the check
		 * to avoid user confusion.
		 */
		if (!$this->hasAutoPrepend)
		{
			$this->auto_prepend['checked']  = '';
			$this->auto_prepend['disabled'] = 'disabled="disabled"';
		}

		if (!$model->hasHtpasswd())
		{
			$this->removeHtpasswdOptions['disabled'] = 'disabled="disabled"';
			$this->removeHtpasswdOptions['checked']  = '';
			$this->removeHtpasswdOptions['help']     = 'SETUP_LBL_SERVERCONFIG_NONEED_HELP';
		}

		$this->hasHtaccess            = $model->hasHtaccess();
		$this->htaccessOptionSelected = 'none';

		$options = ['none'];

		if ($model->hasAddHandler())
		{
			$options[] = 'removehandler';

			$this->htaccessOptionSelected = $model->isNewhost() ? 'removehandler' : 'none';
		}

		if ($model->hasAddHandler())
		{
			$options[] = 'replacehandler';

			$this->htaccessOptionSelected = 'replacehandler';
		}

		$this->htaccessOptionSelected = $model->getState('htaccessHandling', $this->htaccessOptionSelected);
		$selectHelper                 = new Select($this->getContainer());

		foreach ($options as $opt)
		{
			$this->htaccessOptions[] = $selectHelper->option($opt, $lang->text('SETUP_LBL_HTACCESSCHANGE_' . $opt));
		}

		// JavaScript
		$doc->addMediaScript('setup.js');

		// Help URL
		$doc->setHelpURL('https://www.akeeba.com/documentation/brs/wordpress-setup.html');

		// Buttons
		$doc->appendButton('GENERAL_BTN_INLINE_HELP', 'btn-outline-info', 'fa-info-circle', 'show-help');
		$this->addButtonPreviousStep();

		$this->addButtonSubmitStep();

		// Next step URL
		$doc->addScriptOptions(
			'nextStep.url', $this->getContainer()->get('steps')->nextStep()->getUri()->toString()
		);

		// Pass Super Users information to the frontend
		$doc->addScriptOptions(
			'brs.setup', [
				'superusers' => ($this->stateVars->superusers ?? null) ?: [],
			]
		);

		return true;
	}
}